/*
	This is a part of the source code for Pro/DESKTOP.
	Copyright (C) 1998-1999 Parametric Technology Corporation.
	All rights reserved.
*/

//	CircularDuplicate.cpp

#include <stdafx.h>

#include "CUtility.h"
#include "CEdit.h"
#include "CApplication.h"

HRESULT CircularDuplicate(long number, double angle, BOOL isTotalAngle, ISet *objectSet1)
{

	/*
		Creates an empty aObjectSet and copies the contents of the objectSet
		that is passed to this function.
		
		Calculates the separation angle

		Calls the DuplicateRotate helper function iteratively to create
		the duplicate objects. After creating the duplicate objects they
		are auto constrained and the selection is set to the objectSet
	*/

	START_METHOD("Circular Duplicate")

	HRESULT status = CONV_SUCCESS;

	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument *activePart = NULL;
	IWorkplane *activeWorkplane = NULL;
	ISketch *activeSketch = NULL;

	status = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	CHECK_RETURN_STATUS(status)

	// Create an empty set of lines
	ISet *aLineSet1 = NULL;
	status = (GetCLASS(ObjectSet))->CreateAObjectSet(&aLineSet1);
	CHECK_RETURN_STATUS(status)

	long numberOfObjects = 0;
	status = objectSet1->GetCount(&numberOfObjects);
	CHECK_RETURN_STATUS(status)

	// Getting the Iterator
	IIt *pIt = NULL;
	status = (GetCLASS(It))->CreateAObjectIt(objectSet1,&pIt);
	CHECK_RETURN_STATUS(status)

	ISet *objectSet2 = NULL;

	IDispatch *pDisp = NULL;
	pIt->start(&pDisp);
	while (pDisp) {
		IGeometric *geometric1 = CAST(IGeometric, pDisp);

		status = aLineSet1->AddMember(CAST(IObjectOrSet, geometric1));
		CHECK_RETURN_STATUS(status)
		pDisp = NULL;
		pIt->Next(&pDisp);
	}

	objectSet2 = aLineSet1;

	double separationAngle = 0;

	// calculate the separation angle
	if( isTotalAngle )
		separationAngle = angle / number;
	else
		separationAngle = angle;

	for( int i=0; i<number-1; i++ )
		DuplicateRotate(separationAngle, objectSet1, objectSet2);

	// Apply AutoConstrain on the set of lines
	status = activeWorkplane->AutoConstrain(CAST(IObjectOrSet, objectSet2));
	CHECK_RETURN_STATUS(status)

	// Set the selection to the object set
	status = activeGraphicDoc->SetSelection(CAST(IObjectOrSet, objectSet2));
	CHECK_RETURN_STATUS(status)

	END_METHOD("Circular Duplicate")
}

HRESULT DuplicateRotate(double rotationAngle, ISet *objectSet1, ISet *objectSet2)
{

	/*
		Helper function for CircularDuplicate()
		Creates duplicates of all objects in the set and adds the newly
		created objects to the existing set of objects
	*/

	START_METHOD("DuplicateRotate")

	HRESULT status = CONV_SUCCESS;
	
	double theta, cosTheta, sinTheta;
	long numberOfObjects = 0;

	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument *activePart = NULL;
	IWorkplane *activeWorkplane = NULL;
	ISketch *activeSketch = NULL;

	status = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	CHECK_RETURN_STATUS(status)

	theta = rotationAngle * PI / 180;
	cosTheta = cos(theta);
	sinTheta = sin(theta);

	IDirection *dir1 = NULL;
	status = (GetCLASS(Direction))->CreateDirection(cosTheta, -sinTheta, 0, &dir1);
	CHECK_RETURN_STATUS(status)

	IDirection *dir2 = NULL;
	status = (GetCLASS(Direction))->CreateDirection(sinTheta, cosTheta, 0, &dir2);
	CHECK_RETURN_STATUS(status)

	IMatrix *rotationMatrix = NULL;
	status = (GetCLASS(Matrix))->CreateRotationMatrix(dir1, dir2, &rotationMatrix);
	CHECK_RETURN_STATUS(status)

	status = objectSet1->GetCount(&numberOfObjects);
	CHECK_RETURN_STATUS(status)

	// Create an empty set of lines
	ISet *tempobjSet1 = NULL;
	status = (GetCLASS(ObjectSet))->CreateAObjectSet(&tempobjSet1);
	CHECK_RETURN_STATUS(status)

	IIt *pIt;
	status = (GetCLASS(It))->CreateAObjectIt(objectSet1,&pIt);
	CHECK_RETURN_STATUS(status)

	IDispatch *pDisp = NULL;
	pIt->start(&pDisp);
	while (pDisp) {
		IGeometric *geometric1 = CAST(IGeometric, pDisp);

		IGeometry *geometry1 = NULL;
		status = geometric1->GetGeometry(&geometry1);
		CHECK_RETURN_STATUS(status)

		IGeometry *geometry2 = NULL;
		status = geometry1->Clone(&geometry2);
		CHECK_RETURN_STATUS(status)

		ILine *aLine1 = NULL;
		status = activeSketch->CreateLine(CAST(ICurve, geometry2), &aLine1);
		CHECK_RETURN_STATUS(status)

		status = tempobjSet1->AddMember(aLine1);
		CHECK_RETURN_STATUS(status)

		status = objectSet2->AddMember(aLine1);
		CHECK_RETURN_STATUS(status)

		status = activeWorkplane->TransformObjects(CAST(IObjectOrSet, geometric1), rotationMatrix);
		CHECK_RETURN_STATUS(status)

		pDisp = NULL;
		pIt->Next(&pDisp);
	}

	END_METHOD("DuplicateRotate")
}
